//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.1
import QtGraphicalEffects 1.0

import wesual.Controls  1.0
import wesual.data.Core 1.0

FocusScope {
    id : documentChooser

    property list<Document> documents
    property string displayedProperty : "label"

    QtObject {
        id : p_

        property int currentIndex : {
            for (var i = 0; i < documents.length; ++i) {
                if (documents[i] === currentDocument)
                    break;
            }
            return i;
        }
    }

    readonly property bool open : popup.popupVisible
    property Document currentDocument : null

    onOpenChanged : {
        if (!open)
            documentChooser.forceActiveFocus();
    }

    function openPopup()  {
        popup.activeIndex = p_.currentIndex;
        popup.show();
        popup.popupItem.focus = true;
    }

    // Key handling
    Keys.onSpacePressed : {
        if (!open) {
            documentChooser.openPopup();
        }
    }
    Keys.onDownPressed : {
        if (!open)
            documentChooser.openPopup();
    }

    width : 100
    implicitHeight : 26
    z : popup.popupVisible ? 10000 : 0
    activeFocusOnTab : true
    focus : true

    Rectangle {
        id : background

        z : 1
        anchors.fill : parent
        color : UI.color(UI.PrimaryBase)

        Behavior on color {
            ColorAnimation { duration : 0 }
        }
        border {
            width : 1
            color : UI.color(UI.PrimaryControlBorder)
        }

        states : [
            State {
                name : "disabled"
                when : !documentChooser.enabled

                changes : [
                    PropertyChanges {
                        target : background
                        color  : UI.color(UI.PrimaryBase)
                    },
                    PropertyChanges {
                        target : indicator
                        color  : UI.color(UI.DisabledButton)
                    },
                    PropertyChanges {
                        target : textShown
                        color  : UI.color(UI.DisabledText)
                    },
                    PropertyChanges {
                        target : background.border
                        color  : UI.color(UI.DisabledBorder)
                    }
                ]
            },
            State {
                name : "selected"
                when : mouseArea.pressed || documentChooser.open

                changes : [
                    PropertyChanges {
                        target : background
                        color : UI.color(UI.PrimaryPress)
                    },
                    PropertyChanges {
                        target : indicator
                        color  : UI.color(UI.PrimaryBase)
                    },
                    PropertyChanges {
                        target : textShown
                        color  : UI.color(UI.SelectedTextColor)
                    },
                    PropertyChanges {
                        target : background.border
                        color  : UI.color(UI.ControlBorderPress)
                    }
                ]
            },
            State {
                name : "hovered"
                when : mouseArea.containsMouse || documentChooser.activeFocus

                changes : [
                    PropertyChanges {
                        target : background
                        color  : UI.color(UI.PrimaryHover)
                    },
                    PropertyChanges {
                        target : background.border
                        color  : UI.color(UI.ControlBorderHover)
                    }
                ]
            }
        ]

        Behavior on border.color {
            ColorAnimation { duration : 0 }
        }
        Text {
            id : textShown
            anchors {
                fill : parent
                leftMargin  : 6
                rightMargin : indicator.width
            }
            text  : currentDocument ? currentDocument[displayedProperty] : ""
            elide : Text.ElideRight
            font {
                family : UI.fontFamily(UI.PrimaryFont)
                weight : UI.fontWeight(UI.PrimaryFont)
                pixelSize : 14
            }
            verticalAlignment : Qt.AlignVCenter
        }

        ColorizedImage {
            id : indicator
            anchors {
                right : parent.right
                rightMargin : 6
                verticalCenter : parent.verticalCenter
                verticalCenterOffset : 1
            }
            source : "qrc:/controls/icons/dropdown-indicator.png"
            color : UI.color(UI.DropdownIndicatorIcon)
        }
    }

    MouseArea {
        id : mouseArea

        z : 2
        hoverEnabled : true
        anchors.fill : parent
        acceptedButtons : Qt.LeftButton
        onPressed : documentChooser.openPopup()
        enabled : documentChooser.enabled
    }

    DocumentChooserPopup {
        id : popup

        z : 0

        anchors.top : parent.bottom

        documents : documentChooser.documents
        displayedProperty : documentChooser.displayedProperty
        documentChooser : documentChooser
        currentSelection : p_.currentIndex
        onAccepted : {
            documentChooser.currentDocument = documents[result];
            popup.hide();
        }
    }
}
